/**
 * prodcons.c
 * 
 * Дана програма моделює задачу виробників і споживачів на базі
 * паралельних потоків. 
 */

#include <assert.h>
#include <errno.h>
#include <pthread.h>
#include <semaphore.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

enum {
        BUFSIZE = 10000,        /* Кількість буферів у буферному пулі */
        NPRODUCERS = 2,         /* Кількість виробників */
        NCONSUMERS = 2,         /* Кількість споживачів */
        NITEMS = 1000000        /* Кількість порцій даних, які виробники
                                   повинні виробити, а споживачі - спожити */
};

/* Буферний пул (кожен буфер - змінна типу int) */
int buf[BUFSIZE];
/* Номери, відповідно, першого порожнього і першого заповненого буферів
   у пулі */
int next_empty, next_full;
/* Семафори, які служать лічильниками, відповідно, порожніх і заповнених
   буферів */
sem_t empty_sem, full_sem;
/* Семафори, які служать лічильниками, відповідно, вироблених і спожитих
   порцій даних */
sem_t produced_total, consumed_total;
/* Масиви для обліку порцій даних, відповідно, вироблених кожним виробником
   і спожитих від кожного виробника (при відсутності змагань повинні
   збігатись) */
int produced_ind[NPRODUCERS], consumed_ind[NPRODUCERS];

void *producer(void *), *consumer(void *);


int main(int argc, char **argv)
{
        pthread_t producers[NPRODUCERS];
        pthread_t consumers[NCONSUMERS];
        int producer_args[NPRODUCERS];
        int i, rval;

        /* Ініціалізується. */
        if (sem_init(&empty_sem, 0, BUFSIZE) == -1) {
                fprintf(stderr, "Error initializing semaphore: %s\n",
                                                        strerror(errno));
                exit(EXIT_FAILURE);
        }
        if (sem_init(&full_sem, 0, 0) == -1) {
                fprintf(stderr, "Error initializing semaphore: %s\n",
                                                        strerror(errno));
                exit(EXIT_FAILURE);
        }
        if (sem_init(&consumed_total, 0, NITEMS) == -1) {
                fprintf(stderr, "Error initializing semaphore: %s\n",
                                                        strerror(errno));
                exit(EXIT_FAILURE);
        }
        if (sem_init(&produced_total, 0, NITEMS) == -1) {
                fprintf(stderr, "Error initializing semaphore: %s\n",
                                                        strerror(errno));
                exit(EXIT_FAILURE);
        }
        next_empty = 0;
        next_full = 0;
        for (i = 0; i < NPRODUCERS; i++) {
                produced_ind[i] = 0;
                consumed_ind[i] = 0;
        }

        /* Створює робочі потоки. */
        for (i = 0; i < NPRODUCERS; i++) {
                producer_args[i] = i;
                rval = pthread_create(&producers[i], NULL, producer,
                                                        &producer_args[i]);
                if (rval != 0) {
                        fprintf(stderr, "Error creating thread: %s\n",
                                                        strerror(rval));
                        exit(EXIT_FAILURE);
                }
        }
        for (i = 0; i < NCONSUMERS; i++) {
                rval = pthread_create(&consumers[i], NULL, consumer, NULL);
                if (rval != 0) {
                        fprintf(stderr, "Error creating thread: %s\n",
                                                        strerror(rval));
                        exit(EXIT_FAILURE);
                }
        }
        /* Чекає завершення робочих потоків. */
        for (i = 0; i < NPRODUCERS; i++) {
                rval = pthread_join(producers[i], NULL);
                assert(rval == 0);
        }
        for (i = 0; i < NCONSUMERS; i++) {
                rval = pthread_join(consumers[i], NULL);
                assert(rval == 0);
        }

        /* Аналізує результати. */
        {
                int consumed, produced;
                int errnum = 0;
	
                for (i = 0; i < NPRODUCERS; i++) {
                        printf("produced_ind[%d] = %d\t\t"
                                "consumed_ind[%d] = %d",
                                i, produced_ind[i], i, consumed_ind[i]);
                        if (produced_ind[i] != consumed_ind[i]) {
                                errnum++;
                                printf("\tERROR\n");
                        } else
                                printf("\tOK\n");
                }
                sem_getvalue(&produced_total, &produced);
                printf("Produced together: %d\n", NITEMS - produced);
                sem_getvalue(&consumed_total, &consumed);
                printf("Consumed together: %d\n", NITEMS - consumed);
                if (errnum)
                        printf("There are some errors\n");
                else
                        printf("There are no errors\n");	
        }

        /* Деініціалізується. */
        rval = sem_destroy(&empty_sem);
        assert(rval == 0);
        rval = sem_destroy(&full_sem);
        assert(rval == 0);
        rval = sem_destroy(&produced_total);
        assert(rval == 0);
        rval = sem_destroy(&consumed_total);
        assert(rval == 0);

        exit(EXIT_SUCCESS);
}

/* Головна функція виробника */
void *producer(void *arg)
{
        int number, rval;

        number =  *((int *) arg);
        rval = 0;

        for (;;) {
                /* Якщо роботи більше немає, завершується. */
                if (sem_trywait(&produced_total) == -1)
                        return((void *) rval);
                /* Якщо немає порожніх буферів, чекає. */
                rval = sem_wait(&empty_sem);
                assert(rval == 0);
                /* Виробляє порцію даних (просто записує в перший вільний
                   буфер свій номер). */
                buf[next_empty] = number;
                next_empty++;
                next_empty %= BUFSIZE;
                produced_ind[number]++;
                /* Збільшує лічильник заповнених буферів. */
                rval = sem_post(&full_sem);
                assert(rval == 0);
        }
        return((void *) rval);
}

/* Головна функція споживача */
void *consumer(void *arg)
{
        int rval;

        rval = 0;

        for (;;) {
                /* Якщо роботи більше немає, завершується. */
                if (sem_trywait(&consumed_total) == -1)
                        return((void *) rval);
                /* Якщо немає заповнених буферів, чекає. */
                rval = sem_wait(&full_sem);
                assert(rval == 0);
                /* Споживає порцію даних. */
                consumed_ind[buf[next_full]]++;
                next_full++;
                next_full %= BUFSIZE;
                /* Збільшує лічильник порожніх буферів. */
                rval = sem_post(&empty_sem);
                assert(rval == 0);
        }
        return((void *) rval);
}
